/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes
#include <cmath>
#include <algorithm>
#include <limits> // for std::numeric_limits


/////////////////////// Qt includes
#include <QPushButton>
#include <QTimer>
#include <QSettings>


/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/MassPeakShaperConfig.hpp"

#include "MsXpS/libXpertMassGui/ToleranceWidget.hpp"

#include "ui_ToleranceWidget.h"

namespace MsXpS
{

namespace libXpertMassGui
{


/*!
\class MsXpS::libXpertMassGui::ToleranceWidget
\inmodule libXpertMassGui
\ingroup XpertMassGuiMassCalculations
\inheaderfile ToleranceWidget.hpp

\brief The ToleranceWidget class provides a widget for the
configuration of the mass tolerance.

This composite widget contains all the widgets and all the logic required to
fully configure the mass peak tolerance.
*/

/*!
\variable MsXpS::libXpertMassGui::ToleranceWidget::mp_ui

\brief The graphical user interface definition.
*/

/*!
\brief Constructs a ToleranceWidget instance.

\list
\li \a parent_p: the parent widget.
\li \a config_settings_file_path: the QSettings configuration file path
\li \a nominal: the nominal libXpertMassCore::Tolerance value
\li \a type: the libXpertMassCore::Tolerance::Type type
\endlist
*/
ToleranceWidget::ToleranceWidget(QWidget *parent_p,
                                 const QString &config_settings_file_path,
                                 double nominal,
                                 libXpertMassCore::Tolerance::Type type)
  : QWidget(parent_p),
    m_configSettingsFilePath(config_settings_file_path),
    mp_tolerance(new libXpertMassCore::Tolerance(this)),
    mp_ui(new Ui::ToleranceWidget)
{
  qDebug() << "Creating ToleranceWidget";

  if(!parent_p)
    qFatal("Programming error. Program aborted.");

  mp_ui->setupUi(this);
  mp_ui->nominalDoubleSpinBox->setRange(0, 2000000);

  setupWidget(nominal, type);
}


/*!
\brief Constructs a ToleranceWidget instance.

\list
\li \a parent_p: the parent widget.
\li \a config: the configuration of the mass peak shaping process
\endlist
*/
ToleranceWidget::ToleranceWidget(QWidget *parent_p,
                                 const QString &config_settings_file_path)
  : QWidget(parent_p),
    m_configSettingsFilePath(config_settings_file_path),
    mp_tolerance(new libXpertMassCore::Tolerance(this)),
    mp_ui(new Ui::ToleranceWidget)
{
  qDebug() << "Creating ToleranceWidget";

  if(!parent_p)
    qFatal("Programming error. Program aborted.");

  mp_ui->setupUi(this);
  mp_ui->nominalDoubleSpinBox->setRange(0, 2000000);

  setupWidget();
}

/*!
\brief Destructs this ToleranceWidget instance.
*/
ToleranceWidget::~ToleranceWidget()
{
  writeSettings(m_configSettingsFilePath);

  delete mp_ui;
}


/*!
\brief Writes the settings of this widget to \a config_settings_file_path for
later restoration.
*/
void
ToleranceWidget::writeSettings(const QString &config_settings_file_path)
{
  QSettings settings(config_settings_file_path, QSettings::IniFormat);

  settings.beginGroup("ToleranceWidget");

  settings.setValue("nominal", mp_tolerance->getNominal());
  libXpertMassCore::Tolerance::Type type =
    mp_tolerance->getType(mp_ui->typeComboBox->currentText());
  settings.setValue("type", static_cast<int>(type));

  settings.endGroup();
}

/*!
\brief Sets the Tolerance nominal value to \a nominal.
*/
void
ToleranceWidget::setNominal(double nominal)
{
  if(nominal != mp_tolerance->getNominal())
    mp_tolerance->setNominal(nominal);
  emit nominalChangedSignal();
}

/*!
\brief Returns the Tolerance nominal value.
*/
double
ToleranceWidget::getNominal() const
{
  return mp_tolerance->getNominal();
}

/*!
\brief Sets the Tolerance type to \a type.
*/
void
ToleranceWidget::setType(libXpertMassCore::Tolerance::Type type)
{
  if(type != mp_tolerance->getType())
    mp_tolerance->setType(type);
  emit typeChangedSignal();
}

/*!
\brief Returns the Tolerance type .
*/
libXpertMassCore::Tolerance::Type
ToleranceWidget::getType() const
{
  return mp_tolerance->getType();
}


/*!
\brief Initializes the libXpertMassCore::Tolerance member using \a nominal and \a
type.
*/
void
ToleranceWidget::initializeTolerance(double nominal,
                                     libXpertMassCore::Tolerance::Type type)
{
  mp_tolerance->setNominal(nominal);
  mp_tolerance->setType(type);
}


/*!
\brief Initializes the libXpertMassCore::Tolerance member using \a nominal and \a
type.
*/
void
ToleranceWidget::initializeTolerance(const libXpertMassCore::Tolerance &tolerance)
{
  mp_tolerance->initialize(tolerance);
}


/*!
\brief Initializes the libXpertMassCore::Tolerance member using values read from the
QSettings settings.

If the settings are not found, the libXpertMassCore::Tolerance member is left
untouched and has thus default values set in the class declaration.

Returns true if the settings were found, false otherwise.
*/
bool
ToleranceWidget::initializeToleranceFromSettings()
{
  bool settings_found = true;

  QSettings settings(m_configSettingsFilePath, QSettings::IniFormat);

  settings.beginGroup("ToleranceWidget");

  double nominal;
  libXpertMassCore::Tolerance::Type type;

  if(!settings.contains("nominal"))
    settings_found = false;
  else
    nominal = settings.value("nominal").toDouble();

  if(!settings.contains("type"))
    settings_found = false;
  else
    type = static_cast<libXpertMassCore::Tolerance::Type>(
      settings.value("type").toInt());

  if(!settings_found)
    return false;

  mp_tolerance->setNominal(nominal);
  mp_tolerance->setType(type);

  return true;
}


/*!
\brief Sets this widget up for the common data.
*/
void
ToleranceWidget::setupWidgetCommon()
{
  mp_ui->nominalDoubleSpinBox->setValue(mp_tolerance->getNominal());
  mp_ui->typeComboBox->setCurrentText(mp_tolerance->getTypeAsString());

  connect(mp_ui->nominalDoubleSpinBox,
          &QDoubleSpinBox::valueChanged,
          this,
          &ToleranceWidget::nominalDoubleSpinBoxValueChanged);

  connect(mp_ui->typeComboBox,
          &QComboBox::currentTextChanged,
          this,
          &ToleranceWidget::typeComboBoxValueChanged);
}


/*!
\brief Sets this widget up.

First the libXpertMassCore::Tolerance member is initialized from QSettings settings,
if possible. Then the values are set to the widgets.
*/
void
ToleranceWidget::setupWidget()
{
  initializeToleranceFromSettings();
  setupWidgetCommon();
}


/*!
\brief Sets this widget up using \a nominal and \a type.

The libXpertMassCore::Tolerance member is initialized using the arguments
and the values are set to the widgets.
*/
void
ToleranceWidget::setupWidget(double nominal, libXpertMassCore::Tolerance::Type type)
{
  initializeTolerance(nominal, type);
  setupWidgetCommon();
}

/*!
\brief Signals that the nominal value is now \a value.
*/
void
ToleranceWidget::nominalDoubleSpinBoxValueChanged(double value)
{
  mp_tolerance->setNominal(value);
  emit nominalChangedSignal();
}

/*!
\brief Signals that the Tolerance type is now \a type.
*/
void
ToleranceWidget::typeComboBoxValueChanged(const QString &text)
{
  mp_tolerance->setType(text);
  emit typeChangedSignal();
}


} // namespace libXpertMassGui

} // namespace MsXpS
